%% Simulate overcapacity, Chemostat
% Introduce an extra sector which contains proteins that are not necessary
% to obtain the simulated flux profile, and try to allocate as many
% proteins as possible to this sector.
clear all; close all; clc;

v_all =[];
f_all = [];

for loop = 1:1:24

data_crabtree = load('ProteomeOptimization_Crabtree_X.mat');

phi = data_crabtree.fvals(loop,:); %

% Add an extra sector
names = {'Upt', 'Uglc', 'Lglc', 'Ferm', 'ESnk', 'Resp' 'Treh', 'Grwt', 'Struc', 'Extra'};
phi(10) = 0;

% Define conditions
D = data_crabtree.D(loop);             % 1/h
chsttime = 100;     % h

% Chemostat experiment
c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];     % mM
tfeed = 1; tcycle = 1;

% Solve system of ODEs
options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
[t_ref, c_ref] = ode15s(@(t,c) protmdl(t,c,phi,D,tcycle,tfeed), [0:0.01:chsttime], c0, options);

% Retrieve fluxes at all timepoints
v_ref = [];
for i = 1:length(t_ref)
    [dcdt, vs] = protmdl(t_ref(i),c_ref(i,:)',phi,0.4,1,1);
    v_ref = [v_ref; vs'];
end

%% Overcapacity simulation
f_seed = phi;

% Start with overcapacity in the respiration & uptake sectors, as a result
% of sector sizes that increase the maximum capacity.
f_upt_max = 0.016;
f_mit_max = 0.12;
Cscrit = 0.057;

if f_seed(1) > f_upt_max
    f_seed(10) = f_seed(10) + (f_seed(1) - f_upt_max);
    f_seed(1)  = f_upt_max;
end

if f_seed(6) > f_mit_max
    f_seed(10) = f_seed(10) + (f_seed(6) - f_mit_max);
    f_seed(6)  = f_mit_max;
end

%remove ethanol sector if below critical glucose concentration
if c_ref(end,1) < Cscrit
    f_seed(10) = f_seed(10) + f_seed(4);
    f_seed(4)  = 0;
end

% Reduce the structure sector to the minimal required value at this
% residual glucose concentration. The sector is not adapted if the 
% simulated cells already receive a penalty.
f_mtn_req = 0.0203/(1+(c_ref(end,1)/0.02)^21)+0.0618;
if f_mtn_req < f_seed(9)
    f_seed(10) = f_seed(10) + (f_seed(9) - f_mtn_req);
    f_seed(9)  = f_mtn_req;
end

% Initial guess based on previous simulations
f_seed(7) = 0;
f_seed(10) = 1 - sum(f_seed(1:9));

% Update optimized proteome
f_opt = f_seed;

% Start with the biggest sector
f_sort = f_opt(1:8);
[f_sort, sortindex] = sort(f_sort,'descend');

for s = sortindex
    % Reset switch
    stop = 0;
    
    while stop == 0
        % Decrease the sector size by one percent
        f_seed     = f_opt;
        if f_seed(s) <= 0.0005
            f_seed(10) = f_opt(10) + 1*f_opt(s);
            f_seed(s)  = f_opt(s)*0;
        else
            f_seed(10) = f_opt(10) + 0.01*f_opt(s);
            f_seed(s)  = f_opt(s)*0.99;
        end
        
            
        
        % Solve system of ODEs
        options = odeset('NonNegative',1:numel(c0),'AbsTol',1e-4);
        [t, c] = ode15s(@(t,c) protmdl(t,c,f_seed,D,tcycle,tfeed), [0:0.01:chsttime], c0, options);

        % Retrieve fluxes at all timepoints
        v = [];
        for i = 1:length(t)
            [dcdt, vs] = protmdl(t(i),c(i,:)',f_seed,0.4,1,1);
            v = [v; vs'];
        end
        
        % Check for imbalanced states
        if max(c(end,7)) < 1
            % If phosphate is too low, it must be imbalanced
            unb = 1;
        elseif max(c(end,8)) > 10
            % If intermediates reach high concentrations, there must be imbalance
            unb = 1;
        elseif max(c(end,9)) > 10
            unb = 1;
        elseif max(c(end,10)) > 15
            unb = 1;
        elseif isreal(c) == 0
            unb = 1;
        else
            unb = 0;
        end
        
        cs_t_avg = trapz(t,c(:,1).*t)/trapz(t,t);
        cs_t_avg_ref = trapz(t_ref,c_ref(:,1).*t_ref)/trapz(t_ref,t_ref);
        cs_ratio = cs_t_avg/cs_t_avg_ref;
        
        cx_dif = (c(1,2)-c(end,2))/c(1,2);
        cx_dif_ref = (c_ref(1,2)-c_ref(end,2))/c_ref(1,2);
        cx_ratio = cx_dif/cx_dif_ref;
        
        if unb == 1 || cs_ratio > 1.05 || cx_ratio < 0.95 || f_seed(s) <= 0
            disp(['UPDATE, ', names{s}, ' = ', num2str(f_opt(s))])
            stop = 1;
        else
            f_opt = f_seed;
            max(c(:,10));
            disp([names{s}, ' = ', num2str(f_seed(s))])
            stop = 0;
        end
    end
end

% Correct the structure sector if smaller than required
f_mtn_req = 0.0203/(1+(c(end,1)/0.02)^21)+0.0618;
delta_f = f_mtn_req - f_opt(9);
if delta_f > 0 
    f_opt(10) = f_opt(10) - delta_f;
    f_opt(9) = f_mtn_req;
end

v_all = [v_all;v(end,:)];
f_all = [f_all;f_opt];

end
save('ProteomeOptimization_Crabtree_X_opt.mat')
